﻿#WaaS Info Script CompatScan
#Phase 1 is at start of TS, grabs basic info and writes to registry.
#
#  CompatScan Keys:
#   CompatScanLastRun
#   CompatScanAttempts
#   CompatScanRunTime
#   CompatScanDownloadTime
#   CompatScanReturnStatus
#   CompatScanReturnCode
#   CompatScanFailedStepName
#   CompatScanFailedStepReturnCode
#   CompatScanFailedAttempts
#   WaaS_Stage

#NOTE, CompatScanHardBlock & CompatScanVPN are individual Steps in the TS, which is why they are not here.


#Creates Function to Set how many times the TS Runs
function Set-RegistryValueIncrement {
    [cmdletbinding()]
    param (
        [string] $path,
        [string] $Name
    )

    try { [int]$Value = Get-ItemPropertyValue @PSBoundParameters -ErrorAction SilentlyContinue } catch {}
    Set-ItemProperty @PSBoundParameters -Value ($Value + 1).ToString() 
}

#Connects to TS Environment and Creates (confirms) Registry Stucture in place for the Win10 Upgrade Build.
$tsenv = New-Object -COMObject Microsoft.SMS.TSEnvironment
$tsBuild = $tsenv.Value("SMSTS_Build")
$RegistryVar = "HKLM:\$($tsenv.Value('RegistryPath'))"
$registryPath = "$RegistryVar\$tsbuild"

if ( -not ( test-path $registryPath ) ) { 
    new-item -ItemType directory -path $registryPath -force -erroraction SilentlyContinue | out-null
}

#Gets TS Start Time and Records to Registry
New-ItemProperty -Path $registryPath -Name "CompatScanLastRun" -Value $TSEnv.Value("SMSTS_StartTSTime") -Force

#Increments the ammount of times the Precache CompatScan TS runs
Set-RegistryValueIncrement -Path $registryPath -Name CompatScanAttempts


#Gets the Time in Minutes it takes to run the CompatScan and Writes to Registry
if ($TSEnv.Value("SMSTS_FinishTSTime") -ne $null)
    {
    $Difference = ([datetime]$TSEnv.Value("SMSTS_FinishTSTime")) - ([datetime]$TSEnv.Value("SMSTS_StartTSTime")) 
    $Difference = [math]::Round($Difference.TotalMinutes)
    if ( -not ( test-path $registryPath ) ) {new-item -ItemType directory -path $registryPath -force -erroraction SilentlyContinue | out-null}
    New-ItemProperty -Path $registryPath -Name "CompatScanRunTime" -Value $Difference -force
    }

#Gets the Time in Minutes it takes to Download Cache Items and Writes to Registry
if ($TSEnv.Value("SMSTS_FinishTSDownTime") -ne $null)
    {
    $DifferenceDown = ([datetime]$TSEnv.Value("SMSTS_FinishTSDownTime")) - ([datetime]$TSEnv.Value("SMSTS_StartTSDownTime")) 
    $DifferenceDown = [math]::Round($DifferenceDown.TotalMinutes)
    if ( -not ( test-path $registryPath ) ) {new-item -ItemType directory -path $registryPath -force -erroraction SilentlyContinue | out-null}
    if ((Get-Item -Path $registrypath).getValue("CompatScanDownloadTime") -eq $null) {New-ItemProperty -Path $registryPath -Name "CompatScanDownloadTime" -Value $DifferenceDown -force}
    }

#Gets CompatScan Results and Write Code & Friendly Name to Registry
[int64] $decimalreturncode = $tsenv.Value("_SMSTSOSUpgradeActionReturnCode")
#[int64] $hexreturncode = 0xC1900210
$hexreturncode = "{0:X0}" -f [int64]$decimalreturncode

$WinIPURet = @(
    @{ Err = "C1900210"; Msg = 'No compatibility issues.'}
    @{ Err = "C1900208"; Msg = 'Incompatible apps or drivers.' }
    @{ Err = "C1900204"; Msg = 'Selected migration choice is not available.' }
    @{ Err = "C1900200"; Msg = 'Not eligible for Windows 10.' }
    @{ Err = "C190020E"; Msg = 'Not enough free disk space.' }
    @{ Err = "C1900107"; Msg = 'Unsupported Operating System.' }
    @{ Err = "8024200D"; Msg = 'Update Needs to be Downloaded Again.' }
)

 $ErrorMsg = $winipuret | ? err -eq $hexreturncode  | % Msg
    New-ItemProperty -path $registryPath -Name "CompatScanReturnStatus" -PropertyType String -Value $ErrorMsg -Force
    #New-ItemProperty -Path $registryPath -Name "CompatScanReturnCodeDec" -Value $tsenv.Value("_SMSTSOSUpgradeActionReturnCode") -force
    New-ItemProperty -Path $registryPath -Name "CompatScanReturnCode" -PropertyType String -Value $hexreturncode -force

#Adding key for WaaS Stage
if ( -not ( test-path $registryPath )) {new-item -ItemType directory -path $registryPath -force -erroraction SilentlyContinue | out-null}
if ( $hexreturncode -eq "C1900210") 
    {
    if ((Get-Item -Path $registrypath).getValue("CompatScanHardBlock") -ne $null) {Remove-ItemProperty -Path $registrypath -Name CompatScanHardBlock -ErrorAction SilentlyContinue} 
    New-ItemProperty -Path $registryPath -Name "WaaS_Stage" -Value "Ready_for_Scheduling" -force 
    }
Else 
    {
    New-ItemProperty -Path $registryPath -Name "WaaS_Stage" -Value "Precache_Compat_Scan_Failure" -force
    }

<#If Fails / Success 
1) deletes IPURuntime, IPUSetupTime, IPUBuild
2) Increments Failed Attempts
3) Records FailedStepName & LastFailedCode
 
    #>
#Failure
if ($tsenv.Value("AllStepsSucceded") -eq "False")
    {
    New-ItemProperty -Path $registryPath -Name "CompatScanFailedStepName" -Value $tsenv.Value("FailedStepName") -force
    New-ItemProperty -Path $registryPath -Name "CompatScanFailedStepReturnCode" -Value $tsenv.Value("FailedStepReturnCode") -force
    #Add Info if Check Readiness Failed.- Writes FreeSpace to Registry Key CompatScanDiskFreeSpace
    if ($tsenv.Value("FailedStepName") -eq "Check Readiness")
        {
        Get-WmiObject win32_LogicalDisk -Filter "DeviceID='C:'" | % { $FreeSpace = $_.FreeSpace/1GB -as [int] ; $DiskSize = $_.Size/1GB -as [int] }
        New-ItemProperty -Path $registryPath -Name "CompatScanDiskFreeSpace" -Value $FreeSpace -force
        New-ItemProperty -Path $registryPath -Name "WaaS_Stage" -Value "Precache_CheckReadiness_Failed" -force
        }
    #Sets CSFailedAttempts Key and Increments after each Failure.
    Set-RegistryValueIncrement -Path $registryPath -Name "CompatScanFailedAttempts"
    
    
    <# - This was for tracking a specific issue and making the reporting more helpful in what the actual error was.
    $SMSTSLog = Get-Content -Path "$($tsenv.Value("_SMSTSLogPath"))\SMSTS.log"
    $Pattern = "Failed to updated security on object"
    if (Select-String -InputObject $SMSTSLog -Pattern $Pattern -Quiet)
        {
        $EDriveFailure = $true
        if ($tsenv.Value("FailedStepName") -eq "Downloading Driver Package")
            {
            New-ItemProperty -path $registryPath -Name "CompatScanReturnStatus" -PropertyType String -Value $Pattern -Force
            }
        }
    #>
    }
#Success
Else
    {
    #Delete CompatScanFailedStepName if exist from previous upgrade (So it doesn't mess with reports)
    if ((Get-Item -Path $registrypath).getValue("CompatScanFailedStepName") -ne $null) {Remove-ItemProperty -Path $registrypath -Name CompatScanFailedStepName}
    #Delete CompatScanFailedStepReturnCode if exist from previous upgrade (So it doesn't sku results)
    if ((Get-Item -Path $registrypath).getValue("CompatScanFailedStepReturnCode") -ne $null) {Remove-ItemProperty -Path $registrypath -Name CompatScanFailedStepReturnCode}
    #Delete IPUFailedStepName if exist from previous upgrade (So it doesn't mess with reports)
    if ((Get-Item -Path $registrypath).getValue("IPUFailedStepName") -ne $null) {Remove-ItemProperty -Path $registrypath -Name IPUFailedStepName}
    #Delete IPUFailedStepReturnCode if exist from previous upgrade (So it doesn't sku results)
    if ((Get-Item -Path $registrypath).getValue("IPUFailedStepReturnCode") -ne $null) {Remove-ItemProperty -Path $registrypath -Name IPUFailedStepReturnCode}
    }   
